/**
 * Image Generator - Handles image generation with preview workflow.
 *
 * @since 1.0.0
 * @package AgenticWP
 */

(function($) {
	'use strict';

	const AgenticWPAdmin = window.AgenticWPAdmin = window.AgenticWPAdmin || {};
	const data = () => window.agenticwpImages || {};
	const i18n = key => data().i18n?.[key] || key;
	const MIME_TYPES = { png: 'image/png', jpeg: 'image/jpeg', webp: 'image/webp' };

	// State
	let currentBase64 = null, currentFormat = 'png', isGenerating = false, isSaving = false, isSaved = false;
	let originalPromptText = '', isEnhancing = false;

	// DOM cache
	const $els = {};

	function init() {
		// Cache elements by ID prefix - maps suffix to camelCase property name
		['image-prompt', 'char-count', 'image-size', 'image-quality', 'image-format', 'image-background',
		 'generate-btn', 'save-btn', 'download-btn', 'regenerate-btn', 'discard-btn', 'preview-section',
		 'preview-image', 'history-grid', 'generating-section'].forEach(id => {
			$els[id.replace(/-./g, x => x[1].toUpperCase())] = $('#agenticwp-' + id);
		});
		$els.tipsToggle = $('.agenticwp-tips-toggle');
		$els.tips = $('.agenticwp-prompt-tips');
		$els.enhanceBtn = $('.agenticwp-enhance-btn');
		$els.originalPrompt = $('.agenticwp-original-prompt');
		$els.originalPromptText = $('.agenticwp-original-prompt-text');
		$els.revertBtn = $('.agenticwp-revert-btn');
		$els.enhanceStatus = $('.agenticwp-enhance-status');

		bindEvents();
		updateEnhanceButtonState();
		loadHistory();
	}

	function bindEvents() {
		$els.imagePrompt.on('input', () => {
			$els.charCount.text($els.imagePrompt.val().length.toLocaleString());
			updateEnhanceButtonState();
		});
		$(document).on('click', '.agenticwp-template-card', handleTemplateClick);
		$els.tipsToggle.on('click', e => {
			e.preventDefault();
			const expanded = $els.tipsToggle.attr('aria-expanded') === 'true';
			$els.tipsToggle.attr('aria-expanded', !expanded);
			$els.tips.prop('hidden', expanded);
		});
		$els.imageFormat.on('change', handleFormatChange);
		$els.imageBackground.on('change', validateTransparency);
		$els.generateBtn.on('click', handleGenerate);
		$els.saveBtn.on('click', handleSave);
		$els.downloadBtn.on('click', handleDownload);
		$els.regenerateBtn.on('click', () => {
			const prompt = $els.imagePrompt.val().trim();
			if (!prompt) return showNotice(i18n('promptRequired'), 'error');
			generateImage(prompt);
		});
		$els.discardBtn.on('click', () => {
			hidePreview();
			$('.agenticwp-template-card').removeClass('active');
		});
		$els.enhanceBtn.on('click', enhancePrompt);
		$els.revertBtn.on('click', revertPrompt);
	}

	function handleTemplateClick(e) {
		e.preventDefault();
		const $card = $(this);

		// Update active state and ARIA for accessibility
		$('.agenticwp-template-card')
			.removeClass('active')
			.attr('aria-pressed', 'false');
		$card
			.addClass('active')
			.attr('aria-pressed', 'true');

		const prompt = $card.data('prompt');
		if (prompt) $els.imagePrompt.val(prompt).trigger('input').focus();
		if ($card.data('size')) $els.imageSize.val($card.data('size'));
		if ($card.data('background')) $els.imageBackground.val($card.data('background'));
	}

	function handleFormatChange() {
		const $opt = $els.imageBackground.find('option[value="transparent"]');
		$opt.prop('disabled', $els.imageFormat.val() === 'jpeg');
		validateTransparency();
	}

	function validateTransparency() {
		if ($els.imageBackground.val() === 'transparent' && $els.imageFormat.val() === 'jpeg') {
			$els.imageBackground.val('auto');
			showNotice(i18n('transparentWarn'), 'warning');
		}
	}

	function handleGenerate() {
		const prompt = $els.imagePrompt.val().trim();
		if (!prompt) { showNotice(i18n('promptRequired'), 'error'); $els.imagePrompt.focus(); return; }
		if (!isGenerating) generateImage(prompt);
	}

	function updateEnhanceButtonState() {
		const isEmpty = !$els.imagePrompt.val().trim();
		$els.enhanceBtn.prop('disabled', isEmpty || isEnhancing);
	}

	function enhancePrompt() {
		const prompt = $els.imagePrompt.val().trim();
		if (!prompt || isEnhancing) return;

		isEnhancing = true;
		originalPromptText = prompt;
		setEnhanceButtonLoading(true);
		$els.enhanceStatus.text(i18n('enhancing')).prop('hidden', false).removeClass('is-error is-success');

		$.ajax({
			url: data().ajaxUrl,
			type: 'POST',
			data: { action: 'agenticwp_enhance_prompt', _ajax_nonce: data().nonce, prompt }
		}).done(resp => {
			if (resp?.success && resp.data?.enhanced_prompt) {
				$els.imagePrompt.val(resp.data.enhanced_prompt).trigger('input');
				$els.imagePrompt.addClass('is-enhanced');
				setTimeout(() => $els.imagePrompt.removeClass('is-enhanced'), 600);
				$els.originalPromptText.text(originalPromptText);
				$els.originalPrompt.prop('hidden', false);
				$els.enhanceStatus.text(i18n('enhanced')).addClass('is-success');
				setTimeout(() => $els.enhanceStatus.prop('hidden', true), 2000);
			} else {
				$els.enhanceStatus.text(i18n('enhanceError')).addClass('is-error');
			}
		}).fail((xhr) => {
			const status = xhr.status;
			let errorKey = 'enhanceError';
			if (status === 0) errorKey = 'networkError';
			else if (status === 401) errorKey = 'invalidKeyError';
			else if (status === 429) errorKey = 'rateLimitError';
			else if (status >= 500) errorKey = 'serviceError';
			$els.enhanceStatus.text(i18n(errorKey)).addClass('is-error');
		}).always(() => {
			isEnhancing = false;
			setEnhanceButtonLoading(false);
			updateEnhanceButtonState();
		});
	}

	function revertPrompt() {
		if (!originalPromptText) return;
		$els.imagePrompt.val(originalPromptText).trigger('input');
		$els.originalPrompt.prop('hidden', true);
		$els.enhanceStatus.prop('hidden', true);
		originalPromptText = '';
	}

	function setEnhanceButtonLoading(loading) {
		$els.enhanceBtn.toggleClass('is-enhancing', loading).prop('disabled', loading);
		if (loading) {
			$els.enhanceBtn.html('<span class="dashicons dashicons-update"></span> ' + i18n('enhancing'));
		} else {
			$els.enhanceBtn.html('<span class="dashicons dashicons-superhero"></span> Enhance');
		}
	}

	function generateImage(prompt) {
		isGenerating = true;
		setButtonLoading($els.generateBtn, true, i18n('generating'), 'format-image', 'Generate Image');
		hidePreview();
		showGenerating();

		ajaxRequest('agenticwp_generate_image', {
			prompt, size: $els.imageSize.val(), quality: $els.imageQuality.val(),
			format: $els.imageFormat.val(), background: $els.imageBackground.val()
		}, resp => {
			currentBase64 = resp.b64_json;
			currentFormat = resp.format || 'png';
			hideGenerating();
			showPreview();
		}, () => {
			isGenerating = false;
			hideGenerating();
			setButtonLoading($els.generateBtn, false, i18n('generating'), 'format-image', 'Generate Image');
		});
	}

	function handleSave() {
		if (!currentBase64 || isSaving) return;
		isSaving = true;
		const prompt = $els.imagePrompt.val().trim();
		setButtonLoading($els.saveBtn, true, i18n('saving'), 'media-default', 'Save to Media Library');

		ajaxRequest('agenticwp_save_image', {
			b64_json: currentBase64, filename: generateFilename(prompt), format: currentFormat, prompt
		}, () => {
			isSaved = true;
			loadHistory();
		}, () => {
			isSaving = false;
			setButtonLoading($els.saveBtn, false, i18n('saving'), 'media-default', 'Save to Media Library');
			if (isSaved) $els.saveBtn.prop('disabled', true);
		});
	}

	function handleDownload() {
		if (!currentBase64) return;
		const prompt = $els.imagePrompt.val().trim();
		const filename = generateFilename(prompt) + '.' + currentFormat;
		const mimeType = MIME_TYPES[currentFormat] || MIME_TYPES.png;

		// Convert base64 to blob and trigger download
		const byteCharacters = atob(currentBase64);
		const byteNumbers = new Array(byteCharacters.length);
		for (let i = 0; i < byteCharacters.length; i++) {
			byteNumbers[i] = byteCharacters.charCodeAt(i);
		}
		const byteArray = new Uint8Array(byteNumbers);
		const blob = new Blob([byteArray], { type: mimeType });

		const url = URL.createObjectURL(blob);
		const link = document.createElement('a');
		link.href = url;
		link.download = filename;
		document.body.appendChild(link);
		link.click();
		document.body.removeChild(link);
		URL.revokeObjectURL(url);
	}

	function ajaxRequest(action, requestData, onSuccess, onComplete) {
		$.ajax({
			url: data().ajaxUrl, type: 'POST',
			data: { action, _ajax_nonce: data().nonce, ...requestData }
		}).done(resp => {
			if (resp?.success && resp.data) onSuccess(resp.data);
			else showNotice(getErrorMessage(resp?.data?.code, resp), 'error');
		}).fail(xhr => {
			const resp = xhr.responseJSON || {};
			showNotice(getErrorMessage(resp.data?.code, resp, xhr.status), 'error');
		}).always(onComplete);
	}

	function setButtonLoading($btn, loading, loadingText, icon, defaultText) {
		const isGenerateBtn = $btn.is($els.generateBtn);
		$btn.toggleClass('agenticwp-button-loading', loading).prop('disabled', loading);

		if (loading && isGenerateBtn) {
			$btn.html(`<span class="agenticwp-btn-loader"><span class="agenticwp-btn-orb"></span><span class="agenticwp-btn-ring"></span><span class="agenticwp-btn-ring"></span></span> ${loadingText}`);
		} else if (loading) {
			$btn.html(`<span class="dashicons dashicons-update agenticwp-spin"></span> ${loadingText}`);
		} else {
			$btn.html(`<span class="dashicons dashicons-${icon}"></span> ${defaultText}`);
		}
	}

	function showGenerating() {
		$els.generatingSection.prop('hidden', false);
		// No scroll needed - overlay is visible within generator section
	}

	function hideGenerating() {
		$els.generatingSection.prop('hidden', true);
	}

	function showPreview() {
		if (!currentBase64) return;
		isSaved = false;
		$els.previewImage.attr('src', `data:${MIME_TYPES[currentFormat] || MIME_TYPES.png};base64,${currentBase64}`);
		$els.previewSection.prop('hidden', false);
		$els.saveBtn.add($els.downloadBtn).add($els.regenerateBtn).add($els.discardBtn).prop('disabled', false);
		$els.previewSection[0].scrollIntoView({ behavior: 'smooth', block: 'start' });
	}

	function hidePreview() {
		$els.previewSection.prop('hidden', true);
		$els.previewImage.attr('src', '');
		currentBase64 = null;
	}

	function generateFilename(prompt) {
		const name = prompt.substring(0, 50).toLowerCase()
			.replace(/[^a-z0-9\s-]/g, '').replace(/\s+/g, '-').replace(/-+/g, '-').trim();
		return name || 'ai-generated';
	}

	function loadHistory() {
		ajaxRequest('agenticwp_get_recent_images', { limit: 10 },
			resp => renderHistory(resp.images || []),
			() => {} // No-op on complete
		);
	}

	function renderHistory(images) {
		if (!images.length) {
			$els.historyGrid.html('<p class="agenticwp-history-empty">No recent images found. Generate your first image above!</p>');
			return;
		}
		$els.historyGrid.html(images.map(img =>
			`<a href="${escapeHtml(img.edit_url)}" class="agenticwp-history-item" target="_blank" title="${escapeHtml(img.title)}">` +
			`<img src="${escapeHtml(img.thumbnail)}" alt="${escapeHtml(img.title)}"></a>`
		).join(''));
	}

	function escapeHtml(str) {
		if (!str) return '';
		const div = document.createElement('div');
		div.textContent = str;
		return div.innerHTML;
	}

	function getErrorMessage(code, response, httpStatus) {
		const codeMap = {
			rate_limit: 'rateLimitError', invalid_api_key: 'invalidKeyError', no_api_key: 'invalidKeyError',
			content_moderation: 'moderationError', upload_limit_exceeded: 'uploadLimitError',
			image_too_large: 'uploadLimitError', http_error: 'networkError', api_error: 'serviceError'
		};
		if (code && codeMap[code]) return i18n(codeMap[code]);
		const statusMap = { 429: 'rateLimitError', 401: 'invalidKeyError', 0: 'networkError' };
		if (httpStatus in statusMap) return i18n(statusMap[httpStatus]);
		if (httpStatus >= 500) return i18n('serviceError');
		return response?.data?.message || i18n('error');
	}

	function showNotice(message, type) {
		if (AgenticWPAdmin.utils?.showNotice) { AgenticWPAdmin.utils.showNotice(message, type); return; }

		const $notice = $(`<div class="notice notice-${type} is-dismissible"><p>${escapeHtml(message)}</p>` +
			'<button type="button" class="notice-dismiss"><span class="screen-reader-text">Dismiss</span></button></div>');
		$('.agenticwp-images-header').after($notice);
		$notice.find('.notice-dismiss').on('click', () => $notice.fadeOut(300, function() { $(this).remove(); }));
		if (type !== 'error') setTimeout(() => $notice.fadeOut(300, function() { $(this).remove(); }), 5000);
	}

	$(document).ready(() => $('#agenticwp-image-prompt').length && init());
	AgenticWPAdmin.imagesGenerator = { init };
})(jQuery);
