/**
 * Model Selection
 *
 * Handles AI model selection interface with visual model cards and
 * auto-save functionality.
 *
 * @file
 * @since 1.0.0
 * @package AgenticWP
 * @namespace AgenticWPAdmin.modelSelection
 */

(function($) {
	'use strict';

	const AgenticWPAdmin = window.AgenticWPAdmin = window.AgenticWPAdmin || {};
	const utils = AgenticWPAdmin.utils || {};

	/**
	 * Gets admin data from global object.
	 *
	 * @return {Object} Admin data object.
	 */
	function getAdminData() {
		return window.agenticwpAdmin || {};
	}

	/**
	 * Initializes model selection.
	 */
	function init() {
		bindEvents();
	}

	/**
	 * Binds event handlers.
	 */
	function bindEvents() {
		$(document).on('click', '.agenticwp-model-card', handleModelSelect);
	}

	/**
	 * Handles model card selection.
	 *
	 * @param {jQuery.Event} event - Click event.
	 */
	function handleModelSelect(event) {
		const $card = $(event.currentTarget);
		const $radio = $card.find('input[type="radio"]');
		const modelValue = $radio.val();

		$('.agenticwp-model-card').removeClass('selected');
		$card.addClass('selected');
		$radio.prop('checked', true);

		saveModelSelection(modelValue);
	}

	/**
	 * Saves selected model via AJAX.
	 *
	 * @param {string} model - Selected model value.
	 */
	function saveModelSelection(model) {
		const $selectedCard = $('.agenticwp-model-card.selected');
		const $badge = $selectedCard.find('.agenticwp-model-card-badge');
		const originalBadgeText = $badge.text();
		const adminData = getAdminData();

		$badge.text('Saving...').addClass('saving');

		utils.postAction({
			action: 'agenticwp_save_model_selection',
			nonce: adminData.nonces?.save_model,
			data: { model: model },
			onSuccess: function(response) {
				if (response && response.success) {
					$badge.text('Saved').removeClass('saving').addClass('saved');
					setTimeout(() => {
						$badge.text(originalBadgeText).removeClass('saved');
					}, 2000);
				} else {
					const message = response && response.data && response.data.message ? response.data.message : 'Failed to save model selection';
					const errorCode = response && response.data && response.data.code ? response.data.code : 'unknown';
					utils.logError('model_selection', {
						error: 'save_failed',
						action: 'agenticwp_save_model_selection',
						model: model,
						message: message,
						code: errorCode
					});
					$badge.text('Error').removeClass('saving').addClass('error');
					setTimeout(() => {
						$badge.text(originalBadgeText).removeClass('error');
					}, 3000);
					utils.showNotice(message, 'error');
				}
			},
			onError: function(error) {
				utils.logError('model_selection', {
					error: 'ajax_error',
					action: 'agenticwp_save_model_selection',
					model: model,
					details: error
				});
				$badge.text('Error').removeClass('saving').addClass('error');
				setTimeout(() => {
					$badge.text(originalBadgeText).removeClass('error');
				}, 3000);
				utils.showNotice('Network error: ' + error, 'error');
			}
		});
	}

	AgenticWPAdmin.modelSelection = {
		init: init
	};
})(jQuery);