/**
 * Admin Utilities
 *
 * Shared utility functions for admin interface including AJAX helpers,
 * notice display, and button loading states.
 *
 * @file
 * @since 1.0.0
 * @package AgenticWP
 * @namespace AgenticWPAdmin.utils
 */

(function($) {
	'use strict';

	const AgenticWPAdmin = window.AgenticWPAdmin = window.AgenticWPAdmin || {};
	const ORIGINAL_TEXT_KEY = 'agenticwpOriginalText';

	/**
	 * Displays admin notice.
	 *
	 * @param {string} message - Notice message.
	 * @param {string} type - Notice type ('error' or 'success').
	 */
	function showNotice(message, type) {
		const noticeClass = type === 'error' ? 'notice-error' : 'notice-success';
		const $notice = $(
			'<div class="notice ' + noticeClass + ' is-dismissible">' +
				'<p>' + message + '</p>' +
				'<button type="button" class="notice-dismiss">' +
					'<span class="screen-reader-text">Dismiss this notice.</span>' +
				'</button>' +
			'</div>'
		);

		$('.wrap h1').after($notice);

		$notice.find('.notice-dismiss').on('click', function() {
			$notice.fadeOut(300, function() {
				$(this).remove();
			});
		});

		if (type === 'success') {
			setTimeout(function() {
				$notice.fadeOut(300, function() {
					$(this).remove();
				});
			}, 5000);
		}
	}

	/**
	 * Toggles button loading state.
	 *
	 * @param {jQuery} $button - Button element.
	 * @param {boolean} isLoading - Whether button is loading.
	 * @param {string} loadingText - Optional loading text.
	 */
	function toggleButtonLoading($button, isLoading, loadingText) {
		if (!$button || !$button.length) {
			return;
		}

		if (isLoading) {
			if (!$button.data(ORIGINAL_TEXT_KEY)) {
				$button.data(ORIGINAL_TEXT_KEY, $.trim($button.text()));
			}

			if (loadingText) {
				$button.text(loadingText);
			}

			$button.prop('disabled', true).addClass('agenticwp-button-loading');
		} else {
			const originalText = $button.data(ORIGINAL_TEXT_KEY);
			if (originalText) {
				$button.text(originalText);
				$button.removeData(ORIGINAL_TEXT_KEY);
			}

			$button.prop('disabled', false).removeClass('agenticwp-button-loading');
		}
	}

	/**
	 * Performs AJAX POST action.
	 *
	 * @param {Object} options - AJAX options.
	 * @return {jQuery.jqXHR} jQuery AJAX promise.
	 */
	function postAction(options) {
		const settings = $.extend(
			{
				url: (window.agenticwpAdmin && window.agenticwpAdmin.ajaxUrl) || '',
				method: 'POST',
				data: {},
				nonce: null,
				onSuccess: null,
				onError: null,
				onComplete: null
			},
			options || {}
		);

		const payload = $.extend({}, settings.data, {
			action: settings.action,
			_ajax_nonce: settings.nonce
		});

		return $.ajax({
			url: settings.url,
			type: settings.method,
			data: payload
		})
			.done(function(response) {
				if (typeof settings.onSuccess === 'function') {
					settings.onSuccess(response);
				}
			})
			.fail(function(xhr, status, error) {
				if (typeof settings.onError === 'function') {
					settings.onError(error, xhr, status);
					return;
				}

				showNotice('Network error: ' + error, 'error');
			})
			.always(function() {
				if (typeof settings.onComplete === 'function') {
					settings.onComplete();
				}
			});
	}

	/**
	 * Logs error (placeholder for future implementation).
	 */
	function logError() {
	}

	AgenticWPAdmin.utils = {
		showNotice: showNotice,
		toggleButtonLoading: toggleButtonLoading,
		postAction: postAction,
		logError: logError
	};
})(jQuery);
