/**
 * AgenticWP Toast Notifications
 *
 * Manages toast notification display, lifecycle, animations, and auto-hide behavior
 * for background job completion notifications.
 *
 * @file
 * @since 1.0.0
 * @package AgenticWP
 */

/**
 * Toast notification manager.
 *
 * @class
 * @since 1.0.0
 */
class ToastManager {
	/**
	 * Initializes toast manager.
	 */
	constructor() {
		this.container = null;
		this.activeToasts = new Map();
		this.autoHideTimeout = 5000;
		this.init();
	}

	/**
	 * Initializes toast container.
	 */
	init() {
		this.createContainer();
	}

	/**
	 * Creates or finds toast container element.
	 */
	createContainer() {
		this.container = document.querySelector('.agenticwp-toast-container');

		if (!this.container) {
			this.container = document.createElement('div');
			this.container.className = 'agenticwp-toast-container';
			document.body.appendChild(this.container);
		}
	}

	/**
	 * Shows a toast notification.
	 *
	 * @param {string} message - Toast message.
	 * @param {string} type - Toast type ('info', 'success', 'error', 'warning').
	 * @param {Object} options - Additional options.
	 * @return {string} Toast ID.
	 */
	showToast(message, type = 'info', options = {}) {
		const toastId = this.generateToastId();
		const toastElement = this.createToastElement(message, type, toastId, options);

		this.container.appendChild(toastElement);
		this.activeToasts.set(toastId, {
			element: toastElement,
			timeout: null
		});

		requestAnimationFrame(() => {
			toastElement.classList.add('show');
		});

		if (options.autoHide !== false) {
			const timeout = setTimeout(() => {
				this.removeToast(toastId);
			}, options.duration || this.autoHideTimeout);

			this.activeToasts.get(toastId).timeout = timeout;
		}

		return toastId;
	}

	/**
	 * Creates toast DOM element.
	 *
	 * @param {string} message - Toast message.
	 * @param {string} type - Toast type.
	 * @param {string} toastId - Unique toast ID.
	 * @param {Object} options - Additional options.
	 * @return {HTMLElement} Toast element.
	 */
	createToastElement(message, type, toastId, options = {}) {
		const toast = document.createElement('div');
		toast.className = `agenticwp-toast ${type}`;
		toast.setAttribute('data-toast-id', toastId);
		toast.setAttribute('role', 'alert');
		toast.setAttribute('aria-live', 'polite');

		const content = document.createElement('div');
		content.className = 'agenticwp-toast-content';

		if (options.icon || this.getDefaultIcon(type)) {
			const icon = document.createElement('span');
			icon.className = 'agenticwp-toast-icon';
			icon.textContent = options.icon || this.getDefaultIcon(type);
			content.appendChild(icon);
		}

		const messageEl = document.createElement('span');
		messageEl.className = 'agenticwp-toast-message';
		messageEl.textContent = message;
		content.appendChild(messageEl);

		const closeBtn = document.createElement('button');
		closeBtn.className = 'agenticwp-toast-close';
		closeBtn.innerHTML = '×';
		closeBtn.setAttribute('aria-label', 'Close notification');
		closeBtn.addEventListener('click', () => this.removeToast(toastId));
		content.appendChild(closeBtn);

		toast.appendChild(content);
		return toast;
	}

	/**
	 * Removes a toast notification.
	 *
	 * @param {string} toastId - Toast ID to remove.
	 */
	removeToast(toastId) {
		const toastData = this.activeToasts.get(toastId);
		if (!toastData) return;

		const { element, timeout } = toastData;

		if (timeout) {
			clearTimeout(timeout);
		}

		element.classList.remove('show');

		setTimeout(() => {
			if (element.parentNode) {
				element.parentNode.removeChild(element);
			}
			this.activeToasts.delete(toastId);
		}, 300);
	}

	/**
	 * Clears all active toasts.
	 */
	clearAllToasts() {
		this.activeToasts.forEach((_, toastId) => {
			this.removeToast(toastId);
		});
	}

	/**
	 * Gets default icon for toast type.
	 *
	 * @param {string} type - Toast type.
	 * @return {string} Icon emoji.
	 */
	getDefaultIcon(type) {
		const icons = {
			success: '✅',
			error: '❌',
			info: 'ℹ️',
			warning: '⚠️'
		};
		return icons[type] || '';
	}

	/**
	 * Generates unique toast ID.
	 *
	 * @return {string} Unique toast ID.
	 */
	generateToastId() {
		return 'toast-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9);
	}

	/**
	 * Shows success toast.
	 *
	 * @param {string} message - Success message.
	 * @param {Object} options - Additional options.
	 * @return {string} Toast ID.
	 */
	showSuccess(message, options = {}) {
		return this.showToast(message, 'success', options);
	}

	/**
	 * Shows error toast.
	 *
	 * @param {string} message - Error message.
	 * @param {Object} options - Additional options.
	 * @return {string} Toast ID.
	 */
	showError(message, options = {}) {
		return this.showToast(message, 'error', options);
	}
}

window.AgenticWPToast = new ToastManager();