<?php
/**
 * Action Manager core class for managing AI agent prompts.
 *
 * @package AgenticWP
 */

namespace Agentic_WP;

defined( 'ABSPATH' ) || exit;

use Agentic_WP\Error_Handler;

/**
 * Action Manager core class for AgenticWP plugin.
 * Manages predefined AI agent prompts as shortcuts in the chat interface.
 */
class Action_Manager {

	const ACTIONS_OPTION = 'agentic_wp_actions';

	const CATEGORIES_OPTION = 'agentic_wp_action_categories';

	/**
	 * Initializes the action manager with required dependencies.
	 *
	 * Sets up hooks and registers all available actions for the plugin.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'init_default_actions' ) );
	}

	/**
	 * Initialize default actions if not already set.
	 */
	public function init_default_actions(): void {
		$existing_actions = get_option( self::ACTIONS_OPTION, array() );

		if ( empty( $existing_actions ) ) {
			$default_actions = $this->get_default_actions();
			update_option( self::ACTIONS_OPTION, $default_actions );
		}

		$existing_categories = get_option( self::CATEGORIES_OPTION, array() );

		if ( empty( $existing_categories ) ) {
			$default_categories = $this->get_default_categories();
			update_option( self::CATEGORIES_OPTION, $default_categories );
		}
	}

	/**
	 * Get all actions (default + custom).
	 *
	 * @return array Actions array organized by category.
	 */
	public function get_actions(): array {
		return get_option( self::ACTIONS_OPTION, $this->get_default_actions() );
	}

	/**
	 * Get actions organized by category for UI display.
	 *
	 * @return array Actions organized by category.
	 */
	public function get_actions_by_category(): array {
		$all_actions = $this->get_actions();
		$categorized = array();

		foreach ( $all_actions as $category_key => $category_actions ) {
			if ( ! is_array( $category_actions ) ) {
				continue;
			}

			foreach ( $category_actions as $action_key => $action_data ) {
				$category_name = $action_data['category'] ?? ucfirst( str_replace( '_', ' ', $category_key ) );

				if ( ! isset( $categorized[ $category_name ] ) ) {
					$categorized[ $category_name ] = array();
				}

				$categorized[ $category_name ][ $action_key ] = $action_data;
			}
		}

		return $categorized;
	}

	/**
	 * Add or update a custom action.
	 *
	 * @param string $action_key Action identifier.
	 * @param array  $action_data Action configuration.
	 * @return bool|\WP_Error Success status or WP_Error on failure.
	 */
	public function save_action( string $action_key, array $action_data ) {
		$action_key  = sanitize_key( $action_key );
		$action_data = $this->sanitize_action_data( $action_data );

		if ( empty( $action_key ) ) {
			Error_Handler::log_error(
				'action_manager',
				'Empty action key provided',
				array( 'action_data' => $action_data )
			);
			return new \WP_Error(
				'invalid_action_key',
				__( 'Action key cannot be empty.', 'agenticwp' )
			);
		}

		if ( ! $this->validate_action_data( $action_data ) ) {
			Error_Handler::log_error(
				'action_manager',
				'Invalid action data',
				array(
					'action_key' => $action_key,
					'data'       => $action_data,
				)
			);
			return new \WP_Error(
				'invalid_action_data',
				__( 'Invalid action data. Name and prompt are required.', 'agenticwp' )
			);
		}

		$all_actions  = $this->get_actions();
		$category     = $action_data['category'] ?? 'Custom';
		$category_key = sanitize_key( strtolower( str_replace( ' ', '_', $category ) ) );

		if ( ! isset( $all_actions[ $category_key ] ) ) {
			$all_actions[ $category_key ] = array();
		}

		$all_actions[ $category_key ][ $action_key ] = $action_data;

		$result = update_option( self::ACTIONS_OPTION, $all_actions );

		if ( ! $result ) {
			Error_Handler::log_error(
				'action_manager',
				'Failed to save action to database',
				array( 'action_key' => $action_key )
			);
			return new \WP_Error(
				'db_error',
				__( 'Failed to save action.', 'agenticwp' )
			);
		}

		Error_Handler::debug_log(
			'Action saved successfully',
			array(
				'action_key' => $action_key,
				'category'   => $category_key,
			)
		);

		return true;
	}

	/**
	 * Delete a custom action.
	 *
	 * @param string $action_key Action identifier.
	 * @return bool|\WP_Error Success status or WP_Error on failure.
	 */
	public function delete_action( string $action_key ) {
		$all_actions = $this->get_actions();
		$found       = false;

		foreach ( $all_actions as $category_key => &$category_actions ) {
			if ( isset( $category_actions[ $action_key ] ) ) {
				unset( $category_actions[ $action_key ] );
				$found = true;
				break;
			}
		}

		if ( ! $found ) {
			Error_Handler::log_error(
				'action_manager',
				'Action not found for deletion',
				array( 'action_key' => $action_key )
			);
			return new \WP_Error(
				'action_not_found',
				__( 'Action not found.', 'agenticwp' )
			);
		}

		$result = update_option( self::ACTIONS_OPTION, $all_actions );

		if ( ! $result ) {
			Error_Handler::log_error(
				'action_manager',
				'Failed to delete action from database',
				array( 'action_key' => $action_key )
			);
			return new \WP_Error(
				'db_error',
				__( 'Failed to delete action.', 'agenticwp' )
			);
		}

		Error_Handler::debug_log(
			'Action deleted successfully',
			array( 'action_key' => $action_key )
		);

		return true;
	}


	/**
	 * Get all categories.
	 *
	 * @return array Categories array.
	 */
	public function get_categories(): array {
		return get_option( self::CATEGORIES_OPTION, $this->get_default_categories() );
	}

	/**
	 * Save a category.
	 *
	 * @param string $category_key Category identifier.
	 * @param array  $category_data Category configuration.
	 * @return bool|\WP_Error Success status or WP_Error on failure.
	 */
	public function save_category( string $category_key, array $category_data ) {
		$category_key  = sanitize_key( $category_key );
		$category_data = $this->sanitize_category_data( $category_data );

		if ( empty( $category_key ) ) {
			Error_Handler::log_error(
				'action_manager',
				'Empty category key provided',
				array( 'category_data' => $category_data )
			);
			return new \WP_Error(
				'invalid_category_key',
				__( 'Category key cannot be empty.', 'agenticwp' )
			);
		}

		if ( ! $this->validate_category_data( $category_data ) ) {
			Error_Handler::log_error(
				'action_manager',
				'Invalid category data',
				array(
					'category_key' => $category_key,
					'data'         => $category_data,
				)
			);
			return new \WP_Error(
				'invalid_category_data',
				__( 'Invalid category data. Name is required.', 'agenticwp' )
			);
		}

		$all_categories                  = $this->get_categories();
		$all_categories[ $category_key ] = $category_data;

		$result = update_option( self::CATEGORIES_OPTION, $all_categories );

		if ( ! $result ) {
			Error_Handler::log_error(
				'action_manager',
				'Failed to save category to database',
				array( 'category_key' => $category_key )
			);
			return new \WP_Error(
				'db_error',
				__( 'Failed to save category.', 'agenticwp' )
			);
		}

		Error_Handler::debug_log(
			'Category saved successfully',
			array( 'category_key' => $category_key )
		);

		return true;
	}

	/**
	 * Delete a category and handle orphaned actions.
	 *
	 * @param string $category_key Category identifier.
	 * @return bool|\WP_Error Success status or WP_Error on failure.
	 */
	public function delete_category( string $category_key ) {
		$all_categories = $this->get_categories();

		if ( ! isset( $all_categories[ $category_key ] ) ) {
			Error_Handler::log_error(
				'action_manager',
				'Category not found for deletion',
				array( 'category_key' => $category_key )
			);
			return new \WP_Error(
				'category_not_found',
				__( 'Category not found.', 'agenticwp' )
			);
		}

		if ( ! $all_categories[ $category_key ]['is_custom'] ) {
			Error_Handler::log_error(
				'action_manager',
				'Attempted to delete default category',
				array( 'category_key' => $category_key )
			);
			return new \WP_Error(
				'delete_default_category',
				__( 'Cannot delete default categories.', 'agenticwp' )
			);
		}

		unset( $all_categories[ $category_key ] );

		$this->reassign_orphaned_actions( $category_key );

		$result = update_option( self::CATEGORIES_OPTION, $all_categories );

		if ( ! $result ) {
			Error_Handler::log_error(
				'action_manager',
				'Failed to delete category from database',
				array( 'category_key' => $category_key )
			);
			return new \WP_Error(
				'db_error',
				__( 'Failed to delete category.', 'agenticwp' )
			);
		}

		Error_Handler::debug_log(
			'Category deleted successfully',
			array( 'category_key' => $category_key )
		);

		return true;
	}

	/**
	 * Clear all unused custom categories (categories with 0 shortcuts).
	 *
	 * @return array{success: bool, removed: int, categories: array} Result array with removed count and category keys.
	 */
	public function clear_unused_categories(): array {
		$all_categories = $this->get_categories();
		$all_actions    = $this->get_actions();
		$removed        = array();

		$action_counts = array();
		foreach ( $all_actions as $category_key => $category_actions ) {
			$action_counts[ $category_key ] = count( $category_actions );
		}

		foreach ( $all_categories as $category_key => $category_data ) {
			if ( ! ( $category_data['is_custom'] ?? true ) ) {
				continue;
			}

			$count = $action_counts[ $category_key ] ?? 0;
			if ( $count > 0 ) {
				continue;
			}

			$removed[] = $category_key;
			unset( $all_categories[ $category_key ] );
		}

		if ( empty( $removed ) ) {
			return array(
				'success'    => true,
				'removed'    => 0,
				'categories' => array(),
			);
		}

		$result = update_option( self::CATEGORIES_OPTION, $all_categories );

		if ( ! $result ) {
			Error_Handler::log_error(
				'action_manager',
				'Failed to clear unused categories',
				array( 'attempted_removals' => count( $removed ) )
			);
			return array(
				'success'    => false,
				'removed'    => 0,
				'categories' => array(),
			);
		}

		Error_Handler::debug_log(
			'Cleared unused categories',
			array(
				'removed_count' => count( $removed ),
				'categories'    => $removed,
			)
		);

		return array(
			'success'    => true,
			'removed'    => count( $removed ),
			'categories' => $removed,
		);
	}


	/**
	 * Sanitize action data.
	 *
	 * @param array $data Raw action data.
	 * @return array Sanitized action data.
	 */
	private function sanitize_action_data( array $data ): array {
		return array(
			'name'        => sanitize_text_field( $data['name'] ?? '' ),
			'prompt'      => wp_kses_post( $data['prompt'] ?? '' ),
			'category'    => sanitize_text_field( $data['category'] ?? 'Custom' ),
			'icon'        => sanitize_text_field( $data['icon'] ?? 'admin-generic' ),
			'description' => sanitize_text_field( $data['description'] ?? '' ),
			'is_custom'   => (bool) ( $data['is_custom'] ?? true ),
		);
	}

	/**
	 * Validate action data structure.
	 *
	 * @param array $data Action data to validate.
	 * @return bool Validation result.
	 */
	private function validate_action_data( array $data ): bool {
		if ( empty( $data['name'] ) || empty( $data['prompt'] ) ) {
			return false;
		}

		if ( strlen( $data['prompt'] ) > 500 ) {
			return false;
		}

		return true;
	}

	/**
	 * Sanitize category data.
	 *
	 * @param array $data Raw category data.
	 * @return array Sanitized category data.
	 */
	private function sanitize_category_data( array $data ): array {
		return array(
			'name'      => sanitize_text_field( $data['name'] ?? '' ),
			'is_custom' => (bool) ( $data['is_custom'] ?? true ),
		);
	}

	/**
	 * Validate category data structure.
	 *
	 * @param array $data Category data to validate.
	 * @return bool Validation result.
	 */
	private function validate_category_data( array $data ): bool {
		return ! empty( $data['name'] );
	}

	/**
	 * Reassign orphaned actions to uncategorized.
	 *
	 * @param string $deleted_category_key Deleted category key.
	 */
	private function reassign_orphaned_actions( string $deleted_category_key ): void {
		$all_actions = $this->get_actions();

		if ( isset( $all_actions[ $deleted_category_key ] ) ) {
			$orphaned_actions = $all_actions[ $deleted_category_key ];
			unset( $all_actions[ $deleted_category_key ] );

			if ( ! isset( $all_actions['uncategorized'] ) ) {
				$all_actions['uncategorized'] = array();
			}

			foreach ( $orphaned_actions as $action_key => $action_data ) {
				$action_data['category']                     = 'Uncategorized';
				$all_actions['uncategorized'][ $action_key ] = $action_data;
			}

			$result = update_option( self::ACTIONS_OPTION, $all_actions );

			if ( $result ) {
				Error_Handler::debug_log(
					'Reassigned orphaned actions',
					array(
						'category_key' => $deleted_category_key,
						'count'        => count( $orphaned_actions ),
					)
				);
			} else {
				Error_Handler::log_error(
					'action_manager',
					'Failed to reassign orphaned actions',
					array(
						'category_key' => $deleted_category_key,
						'count'        => count( $orphaned_actions ),
					)
				);
			}
		}
	}

	/**
	 * Get default actions configuration.
	 *
	 * @return array Default actions organized by category.
	 */
	private function get_default_actions(): array {
		return array(
			'content_creation' => array(
				'blog_post_creation'    => array(
					'name'        => __( 'Blog Post Creation', 'agenticwp' ),
					'prompt'      => __( 'Create a comprehensive blog post about [TOPIC]. Structure the content with clear headings, include practical examples, and end with actionable takeaways. Aim for 1500-2000 words with proper SEO optimization.', 'agenticwp' ),
					'category'    => __( 'Content', 'agenticwp' ),
					'icon'        => 'edit',
					'description' => __( 'Full blog post with structured blocks', 'agenticwp' ),
					'is_custom'   => false,
				),
				'landing_page_creation' => array(
					'name'        => __( 'Landing Page Creation', 'agenticwp' ),
					'prompt'      => __( 'Create a marketing-focused landing page for [PRODUCT/SERVICE]. Include compelling headlines, clear value propositions, benefit-focused sections, customer testimonials, and strong call-to-action buttons. Structure for maximum conversion with strategic placement of trust signals.', 'agenticwp' ),
					'category'    => __( 'Content', 'agenticwp' ),
					'icon'        => 'laptop',
					'description' => __( 'Marketing-focused page with CTAs', 'agenticwp' ),
					'is_custom'   => false,
				),
				'product_page_creation' => array(
					'name'        => __( 'Product Page Creation', 'agenticwp' ),
					'prompt'      => __( 'Create an e-commerce product page for [PRODUCT]. Include detailed product descriptions, feature lists, technical specifications, high-quality image placeholders, customer reviews section, and purchase call-to-actions. Focus on conversion optimization and addressing customer objections.', 'agenticwp' ),
					'category'    => __( 'Content', 'agenticwp' ),
					'icon'        => 'products',
					'description' => __( 'E-commerce product with images and descriptions', 'agenticwp' ),
					'is_custom'   => false,
				),
				'tutorial_content'      => array(
					'name'        => __( 'Tutorial Content', 'agenticwp' ),
					'prompt'      => __( 'Create an educational step-by-step tutorial about [TOPIC]. Use clear numbered steps, include helpful tips and warnings, add visual content placeholders, and provide troubleshooting sections. Make it accessible for beginners while comprehensive enough for intermediate users.', 'agenticwp' ),
					'category'    => __( 'Content', 'agenticwp' ),
					'icon'        => 'book-alt',
					'description' => __( 'Educational step-by-step guides', 'agenticwp' ),
					'is_custom'   => false,
				),
				'news_article'          => array(
					'name'        => __( 'News Article', 'agenticwp' ),
					'prompt'      => __( 'Create a news-style article about [TOPIC]. Use an inverted pyramid structure with the most important information first. Include quotes, factual data, and multiple perspectives. Provide proper context for readers.', 'agenticwp' ),
					'category'    => __( 'Content', 'agenticwp' ),
					'icon'        => 'admin-site-alt3',
					'description' => __( 'News-style content with structured format', 'agenticwp' ),
					'is_custom'   => false,
				),
				'faq_content'           => array(
					'name'        => __( 'FAQ Content Creation', 'agenticwp' ),
					'prompt'      => __( 'Create comprehensive FAQ content about [TOPIC]. Generate common questions and detailed answers that address customer concerns and pain points. Structure with clear question headings and informative answers. Focus on providing value while naturally incorporating relevant keywords for SEO benefits.', 'agenticwp' ),
					'category'    => __( 'Content', 'agenticwp' ),
					'icon'        => 'editor-help',
					'description' => __( 'Structured Q&A content for user support', 'agenticwp' ),
					'is_custom'   => false,
				),
			),
			'seo_optimization' => array(
				'meta_description'        => array(
					'name'        => __( 'Meta Description Generation', 'agenticwp' ),
					'prompt'      => __( 'Generate an optimized meta description for "[POST_TITLE]". Analyze the content to identify key topics and benefits. Create a compelling 150-160 character description that includes the primary keyword and encourages clicks.', 'agenticwp' ),
					'category'    => __( 'SEO', 'agenticwp' ),
					'icon'        => 'search',
					'description' => __( 'Create optimized meta descriptions', 'agenticwp' ),
					'is_custom'   => false,
				),
				'internal_linking'        => array(
					'name'        => __( 'Internal Linking Optimization', 'agenticwp' ),
					'prompt'      => __( 'Add strategic internal links to the current content in "[POST_TITLE]". Search for relevant existing posts and pages on the website, then add contextually appropriate internal links with natural anchor text. Limit to 2-3 strategic links that enhance user journey and SEO value based on the following keywords: [KEYWORDS]', 'agenticwp' ),
					'category'    => __( 'SEO', 'agenticwp' ),
					'icon'        => 'admin-links',
					'description' => __( 'Add strategic internal links', 'agenticwp' ),
					'is_custom'   => false,
				),
				'content_seo_enhancement' => array(
					'name'        => __( 'Content SEO Enhancement', 'agenticwp' ),
					'prompt'      => __( 'Optimize the existing content for search engines in "[POST_TITLE]". Improve keyword density, add relevant subheadings, enhance readability, and suggest content additions that would improve search rankings. Focus on natural keyword integration and user experience based on the following keywords: [KEYWORDS]', 'agenticwp' ),
					'category'    => __( 'SEO', 'agenticwp' ),
					'icon'        => 'chart-line',
					'description' => __( 'Optimize existing content for search', 'agenticwp' ),
					'is_custom'   => false,
				),
				'content_clarity_polish'  => array(
					'name'        => __( 'Content Clarity & Polish', 'agenticwp' ),
					'prompt'      => __( 'Improve the clarity, readability, and overall polish of the existing content in "[POST_TITLE]". Enhance sentence flow, eliminate jargon, improve paragraph structure, fix awkward phrasing, and ensure consistent tone throughout. Focus on making complex ideas more accessible while maintaining the original meaning and expertise level. Improve transitions between sections and strengthen the overall narrative flow.', 'agenticwp' ),
					'category'    => __( 'SEO', 'agenticwp' ),
					'icon'        => 'edit-page',
					'description' => __( 'Improve readability and content flow', 'agenticwp' ),
					'is_custom'   => false,
				),
				'alt_text_optimization'   => array(
					'name'        => __( 'Alt Text Optimization', 'agenticwp' ),
					'prompt'      => __( 'Analyze all images in the current content in "[POST_TITLE]" and generate SEO-optimized alt text. Create keyword-rich alt text that accurately describes each image while supporting the content\'s SEO goals. Ensure accessibility compliance and natural keyword integration.', 'agenticwp' ),
					'category'    => __( 'SEO', 'agenticwp' ),
					'icon'        => 'format-image',
					'description' => __( 'Generate SEO-optimized alt text for images', 'agenticwp' ),
					'is_custom'   => false,
				),
				'content_gap_analysis'    => array(
					'name'        => __( 'Content Gap Analysis', 'agenticwp' ),
					'prompt'      => __( 'Analyze the current content and identify content gaps and opportunities in "[POST_TITLE]". Research missing topics, keywords, and content types that competitors may be covering. Suggest new content ideas, identify underserved search queries, and recommend content improvements to capture additional organic traffic.', 'agenticwp' ),
					'category'    => __( 'SEO', 'agenticwp' ),
					'icon'        => 'analytics',
					'description' => __( 'Identify missing content opportunities for SEO', 'agenticwp' ),
					'is_custom'   => false,
				),
			),
		);
	}

	/**
	 * Get default categories configuration.
	 *
	 * @return array Default categories.
	 */
	private function get_default_categories(): array {
		return array(
			'content_creation' => array(
				'name'      => __( 'Content', 'agenticwp' ),
				'is_custom' => false,
			),
			'seo_optimization' => array(
				'name'      => __( 'SEO', 'agenticwp' ),
				'is_custom' => false,
			),
			'uncategorized'    => array(
				'name'      => __( 'Uncategorized', 'agenticwp' ),
				'is_custom' => false,
			),
		);
	}
}
