<?php
/**
 * Schema Meta Box for AgenticWP plugin.
 *
 * @package AgenticWP
 */

namespace Agentic_WP;

defined( 'ABSPATH' ) || exit;

/**
 * Schema meta box for post editor.
 *
 * Provides UI for overriding schema.org properties on posts and pages.
 *
 * @since 1.0.0
 */
class Meta_Box_Schema {

	/**
	 * Registers meta box hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
		add_action( 'save_post', array( $this, 'save_schema_meta' ), 10, 2 );
	}

	/**
	 * Adds schema meta box to post editor.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function add_meta_box() {
		$screens = array( 'post', 'page' );

		foreach ( $screens as $screen ) {
			add_meta_box(
				'agenticwp_schema',
				__( 'AgenticWP Schema Override', 'agenticwp' ),
				array( $this, 'meta_box_callback' ),
				$screen,
				'normal',
				'low'
			);
		}
	}

	/**
	 * Render schema meta box content.
	 *
	 * @param \WP_Post $post Current post object.
	 */
	public function meta_box_callback( $post ) {
		wp_nonce_field( 'agenticwp_schema_nonce', 'agenticwp_schema_nonce' );

		$schema_headline    = get_post_meta( $post->ID, '_agenticwp_schema_headline', true );
		$schema_description = get_post_meta( $post->ID, '_agenticwp_schema_description', true );

		?>
		<div class="agenticwp-schema-meta-box">
			<p class="description">
				<?php esc_html_e( 'Override default schema properties for this post. Leave empty to use automatic values.', 'agenticwp' ); ?>
			</p>

			<div class="agenticwp-schema-field">
				<label for="agenticwp_schema_headline"><?php esc_html_e( 'Schema Headline', 'agenticwp' ); ?></label>
				<input type="text"
						id="agenticwp_schema_headline"
						name="agenticwp_schema_headline"
						value="<?php echo esc_attr( $schema_headline ); ?>"
						class="large-text"
						placeholder="<?php echo esc_attr( get_the_title( $post ) ); ?>" />
				<p class="description"><?php esc_html_e( 'Override the schema headline. Defaults to post title.', 'agenticwp' ); ?></p>
			</div>

			<div class="agenticwp-schema-field">
				<label for="agenticwp_schema_description"><?php esc_html_e( 'Schema Description', 'agenticwp' ); ?></label>
				<textarea id="agenticwp_schema_description"
							name="agenticwp_schema_description"
							rows="3"
							class="large-text"
							placeholder="
							<?php
							$meta_desc = get_post_meta( $post->ID, '_agenticwp_meta_description', true );
							if ( ! empty( $meta_desc ) ) {
								echo esc_attr( $meta_desc );
							} else {
								echo esc_attr( wp_trim_words( get_the_excerpt( $post ), 25 ) );
							}
							?>
							"><?php echo esc_textarea( $schema_description ); ?></textarea>
				<p class="description"><?php esc_html_e( 'Override the schema description. Defaults to meta description or excerpt.', 'agenticwp' ); ?></p>
			</div>
		</div>
		<?php
	}

	/**
	 * Save schema meta box data.
	 *
	 * @param int      $post_id Post ID.
	 * @param \WP_Post $post    Post object.
	 */
	public function save_schema_meta( $post_id, $post ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
		if ( ! isset( $_POST['agenticwp_schema_nonce'] ) ||
			! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['agenticwp_schema_nonce'] ) ), 'agenticwp_schema_nonce' ) ) {
			return;
		}

		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		if ( isset( $_POST['agenticwp_schema_headline'] ) ) {
			$headline = sanitize_text_field( wp_unslash( $_POST['agenticwp_schema_headline'] ) );
			if ( ! empty( $headline ) ) {
				update_post_meta( $post_id, '_agenticwp_schema_headline', $headline );
			} else {
				delete_post_meta( $post_id, '_agenticwp_schema_headline' );
			}
		}

		if ( isset( $_POST['agenticwp_schema_description'] ) ) {
			$description = sanitize_textarea_field( wp_unslash( $_POST['agenticwp_schema_description'] ) );
			if ( ! empty( $description ) ) {
				update_post_meta( $post_id, '_agenticwp_schema_description', $description );
			} else {
				delete_post_meta( $post_id, '_agenticwp_schema_description' );
			}
		}
	}
}