<?php
/**
 * Schema Manager for AgenticWP plugin.
 *
 * @package AgenticWP
 */

namespace Agentic_WP;

defined( 'ABSPATH' ) || exit;

/**
 * Handles SEO schema markup generation and output.
 * Generates JSON-LD structured data for posts, pages, and organization.
 */
class Schema_Manager {

	/**
	 * Initialize schema manager and register hooks.
	 */
	public function __construct() {
		add_action( 'wp_head', array( $this, 'output_schema_markup' ) );
	}

	/**
	 * Output appropriate schema markup based on the current page context.
	 *
	 * @return void
	 */
	public function output_schema_markup() {
		if ( is_single() ) {
			$this->output_blog_posting_schema();
		} elseif ( is_page() ) {
			$this->output_web_page_schema();
		}

		$this->output_organization_schema();
	}

	/**
	 * Get schema headline and description for a post.
	 *
	 * @param \WP_Post $post Post object.
	 * @return array Array with 'headline' and 'description' keys.
	 */
	private function get_post_schema_text( $post ) {
		$schema_headline    = get_post_meta( $post->ID, '_agenticwp_schema_headline', true );
		$schema_description = get_post_meta( $post->ID, '_agenticwp_schema_description', true );

		$headline = ! empty( $schema_headline ) ? $schema_headline : get_the_title( $post );

		if ( ! empty( $schema_description ) ) {
			$description = $schema_description;
		} else {
			$meta_description = get_post_meta( $post->ID, '_agenticwp_meta_description', true );
			$description      = ! empty( $meta_description ) ? $meta_description : wp_trim_words( get_the_excerpt( $post ), 25 );
		}

		return array(
			'headline'    => $headline,
			'description' => $description,
		);
	}

	/**
	 * Generate and output BlogPosting schema for single posts.
	 */
	private function output_blog_posting_schema() {
		global $post;

		if ( ! $post ) {
			return;
		}

		$text = $this->get_post_schema_text( $post );

		$schema = array(
			'@context'      => 'https://schema.org',
			'@type'         => 'BlogPosting',
			'headline'      => $text['headline'],
			'description'   => $text['description'],
			'datePublished' => get_the_date( 'c', $post ),
			'dateModified'  => get_the_modified_date( 'c', $post ),
			'publisher'     => $this->get_organization_schema_data(),
			'url'           => get_permalink( $post ),
		);

		$featured_image_id = get_post_thumbnail_id( $post );
		if ( $featured_image_id ) {
			$image_url = wp_get_attachment_image_url( $featured_image_id, 'full' );
			if ( $image_url ) {
				$schema['image'] = array(
					'@type' => 'ImageObject',
					'url'   => $image_url,
				);
			}
		}

		$this->output_json_ld( $schema );
	}

	/**
	 * Generate and output WebPage schema for pages.
	 */
	private function output_web_page_schema() {
		global $post;

		if ( ! $post ) {
			return;
		}

		$text = $this->get_post_schema_text( $post );

		$schema = array(
			'@context'     => 'https://schema.org',
			'@type'        => 'WebPage',
			'name'         => $text['headline'],
			'description'  => $text['description'],
			'url'          => get_permalink( $post ),
			'dateModified' => get_the_modified_date( 'c', $post ),
			'publisher'    => $this->get_organization_schema_data(),
		);

		$this->output_json_ld( $schema );
	}

	/**
	 * Generate and output Organization schema.
	 */
	private function output_organization_schema() {
		$organization_data = $this->get_organization_schema_data();

		if ( ! empty( $organization_data ) ) {
			$this->output_json_ld( $organization_data );
		}
	}

	/**
	 * Get organization schema data from settings.
	 *
	 * @return array Organization schema data
	 */
	private function get_organization_schema_data() {
		$organization = array(
			'@context' => 'https://schema.org',
			'@type'    => 'Organization',
			'name'     => get_bloginfo( 'name' ),
			'url'      => home_url(),
		);

		$org_name = get_option( 'agentic_wp_organization_name', '' );
		if ( ! empty( $org_name ) ) {
			$organization['name'] = $org_name;
		}

		$org_logo = get_option( 'agentic_wp_organization_logo', '' );
		if ( ! empty( $org_logo ) ) {
			$organization['logo'] = array(
				'@type' => 'ImageObject',
				'url'   => $org_logo,
			);
		}

		$contact_info = array();

		$org_email = get_option( 'agentic_wp_organization_email', '' );
		if ( ! empty( $org_email ) ) {
			$contact_info[] = array(
				'@type'       => 'ContactPoint',
				'contactType' => 'customer service',
				'email'       => $org_email,
			);
		}

		$org_phone = get_option( 'agentic_wp_organization_phone', '' );
		if ( ! empty( $org_phone ) ) {
			$contact_info[] = array(
				'@type'       => 'ContactPoint',
				'contactType' => 'customer service',
				'telephone'   => $org_phone,
			);
		}

		if ( ! empty( $contact_info ) ) {
			$organization['contactPoint'] = $contact_info;
		}

		return $organization;
	}

	/**
	 * Output JSON-LD script tag.
	 *
	 * @param array $schema Schema data to output.
	 */
	private function output_json_ld( $schema ) {
		if ( empty( $schema ) ) {
			return;
		}

		echo '<script type="application/ld+json">' . wp_json_encode( $schema, JSON_UNESCAPED_SLASHES ) . '</script>' . "\n";
	}
}
