<?php
/**
 * Get_block_info tool: concise per-block schema and example.
 *
 * @package AgenticWP
 */

namespace Agentic_WP;

use Agentic_WP\Error_Handler;

defined( 'ABSPATH' ) || exit;

/**
 * Get block info tool for retrieving concise per-block schema and examples.
 *
 * @since 1.0.0
 */
class Tool_Get_Block_Info {
	/**
	 * Returns the OpenAI tool schema for getting block information.
	 *
	 * @since 1.0.0
	 *
	 * @return array Tool schema definition.
	 */
	public static function schema(): array {
		return array(
			'type'        => 'function',
			'name'        => 'get_block_info',
			'description' => 'Get concise schema and minimal example for a block slug.',
			'strict'      => true,
			'parameters'  => array(
				'type'                 => 'object',
				'properties'           => array(
					'slug' => array(
						'type'        => 'string',
						'description' => 'Full block name, e.g. core/cover',
					),
				),
				'required'             => array( 'slug' ),
				'additionalProperties' => false,
			),
		);
	}

	/**
	 * Executes the get block info tool.
	 *
	 * @since 1.0.0
	 *
	 * @param array $args Tool arguments containing block slug.
	 * @return string JSON response or error message.
	 */
	public static function run( array $args ): string {
		$slug = isset( $args['slug'] ) && is_string( $args['slug'] ) ? sanitize_text_field( $args['slug'] ) : '';
		if ( '' === $slug ) {
			Error_Handler::log_error(
				'tool_get_block_info',
				'Missing slug parameter'
			);
			return 'ERROR: missing_slug';
		}
		$info = Block_Catalog::get_block_info( $slug );
		if ( empty( $info ) ) {
			Error_Handler::log_error(
				'tool_get_block_info',
				'Block not found',
				array( 'slug' => $slug )
			);
			return 'ERROR: block_not_found';
		}

		Error_Handler::debug_log(
			'Block info retrieved',
			array( 'slug' => $slug )
		);

		return wp_json_encode( $info );
	}
}
