<?php
/**
 * List_blocks tool: compact discovery of registered blocks.
 *
 * @package AgenticWP
 */

namespace Agentic_WP;

defined( 'ABSPATH' ) || exit;

/**
 * List blocks tool for compact discovery of registered blocks.
 *
 * @since 1.0.0
 */
class Tool_List_Blocks {
	/**
	 * Returns the OpenAI tool schema for listing blocks.
	 *
	 * @since 1.0.0
	 *
	 * @return array Tool schema definition.
	 */
	public static function schema(): array {
		return array(
			'type'        => 'function',
			'name'        => 'list_blocks',
			'description' => 'List registered Gutenberg blocks with optional search/category and pagination. Returns compact cards.',
			'strict'      => true,
			'parameters'  => array(
				'type'                 => 'object',
				'properties'           => array(
					'search'   => array(
						'type'        => 'string',
						'description' => 'Optional search across slug and title.',
					),
					'category' => array(
						'type'        => 'string',
						'description' => 'Optional block category filter.',
					),
					'page'     => array(
						'type'        => 'integer',
						'description' => '1-based page index.',
					),
					'per_page' => array(
						'type'        => 'integer',
						'description' => 'Items per page (1-100).',
					),
				),
				'required'             => array( 'search', 'category', 'page', 'per_page' ),
				'additionalProperties' => false,
			),
		);
	}

	/**
	 * Executes the list blocks tool.
	 *
	 * @since 1.0.0
	 *
	 * @param array $args Tool arguments containing search, category, page, and per_page.
	 * @return string JSON response or error message.
	 */
	public static function run( array $args ): string {
		$search   = isset( $args['search'] ) && is_string( $args['search'] ) ? sanitize_text_field( $args['search'] ) : '';
		$category = isset( $args['category'] ) && is_string( $args['category'] ) ? sanitize_text_field( $args['category'] ) : '';
		$page     = isset( $args['page'] ) && is_int( $args['page'] ) ? (int) $args['page'] : 1;
		$per_page = isset( $args['per_page'] ) && is_int( $args['per_page'] ) ? (int) $args['per_page'] : 20;

		if ( $page < 1 ) {
			return 'ERROR: invalid_page';
		}

		if ( $per_page < 1 || $per_page > 100 ) {
			return 'ERROR: invalid_per_page';
		}

		$result = Block_Catalog::list_blocks(
			array(
				'search'   => $search,
				'category' => $category,
				'page'     => $page,
				'per_page' => $per_page,
			)
		);

		if ( ! is_array( $result ) ) {
			return 'ERROR: catalog_error';
		}

		$json = wp_json_encode( $result );
		if ( false === $json ) {
			return 'ERROR: json_encoding_failed';
		}

		return $json;
	}
}
