/**
 * Collapsible Sections
 *
 * Manages collapsible settings sections with state persistence in localStorage.
 *
 * @file
 * @since 1.0.0
 * @package AgenticWP
 * @namespace AgenticWPAdmin.collapsibleSections
 */

(function($) {
	'use strict';

	window.AgenticWPAdmin = window.AgenticWPAdmin || {};

	/**
	 * Collapsible sections manager object.
	 *
	 * @type {Object}
	 */
	const CollapsibleSections = {
		storageKey: 'agenticwp_collapsed_sections',

		init: function() {
			this.bindEvents();
			this.restoreCollapsedState();
		},

		bindEvents: function() {
			$(document).on('click', '.agenticwp-section-toggle', this.handleToggle.bind(this));
		},

		handleToggle: function(e) {
			e.preventDefault();

			const $button = $(e.currentTarget);
			const $section = $button.closest('.agenticwp-settings-section');
			const sectionId = $section.data('section');

			if (!sectionId) {
				return;
			}

			this.toggleSection($section, sectionId);
		},

		toggleSection: function($section, sectionId) {
			if (!$section || !$section.length || !sectionId) {
				return;
			}

			try {
				const $content = $section.find('.agenticwp-section-content');
				const $header = $section.find('.agenticwp-section-header');
				const $toggleButton = $section.find('.agenticwp-section-toggle');

				const isCollapsed = $section.hasClass('agenticwp-section-collapsed');

				if (isCollapsed) {
					$section.removeClass('agenticwp-section-collapsed');
					if ($header.length) {
						$header.removeClass('agenticwp-mb-0');
					}
					if ($content.length) {
						$content.slideDown(300);
					}
					if ($toggleButton.length) {
						$toggleButton.attr('aria-expanded', 'true');
					}
					this.removeSectionFromStorage(sectionId);
				} else {
					$section.addClass('agenticwp-section-collapsed');
					if ($header.length) {
						$header.addClass('agenticwp-mb-0');
					}
					if ($content.length) {
						$content.slideUp(300);
					}
					if ($toggleButton.length) {
						$toggleButton.attr('aria-expanded', 'false');
					}
					this.addSectionToStorage(sectionId);
				}
			} catch (e) {
			}
		},

		getCollapsedSections: function() {
			try {
				const stored = localStorage.getItem(this.storageKey);
				if (!stored) {
					return [];
				}
				const parsed = JSON.parse(stored);
				return Array.isArray(parsed) ? parsed : [];
			} catch (e) {
				return [];
			}
		},

		setCollapsedSections: function(sections) {
			if (!Array.isArray(sections)) {
				return;
			}

			try {
				localStorage.setItem(this.storageKey, JSON.stringify(sections));
			} catch (e) {
			}
		},

		addSectionToStorage: function(sectionId) {
			const collapsedSections = this.getCollapsedSections();
			if (!collapsedSections.includes(sectionId)) {
				collapsedSections.push(sectionId);
				this.setCollapsedSections(collapsedSections);
			}
		},

		removeSectionFromStorage: function(sectionId) {
			const collapsedSections = this.getCollapsedSections();
			const index = collapsedSections.indexOf(sectionId);
			if (index > -1) {
				collapsedSections.splice(index, 1);
				this.setCollapsedSections(collapsedSections);
			}
		},

		restoreCollapsedState: function() {
			try {
				const collapsedSections = this.getCollapsedSections();

				if (!Array.isArray(collapsedSections)) {
					return;
				}

				collapsedSections.forEach((sectionId) => {
					if (!sectionId || typeof sectionId !== 'string') {
						return;
					}

					const $section = $(`[data-section="${$.escapeSelector(sectionId)}"]`);
					if ($section.length) {
						const $content = $section.find('.agenticwp-section-content');
						const $header = $section.find('.agenticwp-section-header');
						const $toggleButton = $section.find('.agenticwp-section-toggle');

						$section.addClass('agenticwp-section-collapsed');
						if ($header.length) {
							$header.addClass('agenticwp-mb-0');
						}
						if ($content.length) {
							$content.hide();
						}
						if ($toggleButton.length) {
							$toggleButton.attr('aria-expanded', 'false');
						}
					}
				});
			} catch (e) {
			}
		}
	};

	window.AgenticWPAdmin.collapsibleSections = CollapsibleSections;

})(jQuery);