/**
 * Tip Jar functionality.
 *
 * Handles visibility, dismissal, and version-based re-showing of the tip jar.
 *
 * @package AgenticWP
 */

( function () {
	'use strict';

	const TipJar = {
		storageKey: 'agenticwp_tip_jar_dismissed',

		/**
		 * Initialize tip jar functionality.
		 */
		init: function () {
			const tipJars = document.querySelectorAll( '[data-tip-jar]' );
			if ( ! tipJars.length ) {
				return;
			}

			// Show tip jar if not dismissed for current major version.
			if ( ! this.isDismissed() ) {
				tipJars.forEach( ( tipJar ) => {
					tipJar.classList.add( 'agenticwp-tip-jar--visible' );
					this.bindEvents( tipJar );
				} );
			}
		},

		/**
		 * Check if tip jar has been dismissed for the current major version.
		 *
		 * @return {boolean} Whether the tip jar is dismissed.
		 */
		isDismissed: function () {
			try {
				const dismissed = localStorage.getItem( this.storageKey );
				if ( ! dismissed ) {
					return false;
				}
				// Re-show after major version updates.
				return this.isSameMajorVersion( dismissed );
			} catch ( e ) {
				// localStorage not available (private browsing, etc.).
				return false;
			}
		},

		/**
		 * Check if the dismissed version matches the current major version.
		 *
		 * @param {string} dismissedVersion The version when dismissed.
		 * @return {boolean} Whether the major versions match.
		 */
		isSameMajorVersion: function ( dismissedVersion ) {
			const currentVersion =
				window.agenticwp_tip_jar?.version || '1.0.0';
			const dismissedMajor = this.getMajorVersion( dismissedVersion );
			const currentMajor = this.getMajorVersion( currentVersion );
			return dismissedMajor === currentMajor;
		},

		/**
		 * Extract major version number from semver string.
		 *
		 * @param {string} version The full version string.
		 * @return {string} The major version number.
		 */
		getMajorVersion: function ( version ) {
			const parts = version.split( '.' );
			return parts[ 0 ] || '1';
		},

		/**
		 * Dismiss the tip jar.
		 */
		dismiss: function () {
			try {
				const version = window.agenticwp_tip_jar?.version || '1.0.0';
				localStorage.setItem( this.storageKey, version );
			} catch ( e ) {
				// localStorage not available.
			}

			// Hide all tip jars on the page.
			document.querySelectorAll( '[data-tip-jar]' ).forEach( ( tipJar ) => {
				tipJar.classList.remove( 'agenticwp-tip-jar--visible' );
			} );
		},

		/**
		 * Bind event listeners to a tip jar element.
		 *
		 * @param {HTMLElement} tipJar The tip jar element.
		 */
		bindEvents: function ( tipJar ) {
			// Dismiss button click.
			const dismissBtn = tipJar.querySelector(
				'.agenticwp-tip-jar-dismiss'
			);
			if ( dismissBtn ) {
				dismissBtn.addEventListener( 'click', () => this.dismiss() );
			}

			// Escape key dismissal.
			document.addEventListener( 'keydown', ( e ) => {
				if (
					e.key === 'Escape' &&
					tipJar.classList.contains( 'agenticwp-tip-jar--visible' )
				) {
					this.dismiss();
				}
			} );
		},
	};

	// Initialize when DOM is ready.
	if ( document.readyState === 'loading' ) {
		document.addEventListener( 'DOMContentLoaded', () => TipJar.init() );
	} else {
		TipJar.init();
	}
} )();
