/**
 * Admin Tooltips
 *
 * Manages interactive tooltips with smooth fade/slide animations and
 * click-to-toggle behavior.
 *
 * @file
 * @since 1.0.0
 * @package AgenticWP
 * @namespace AgenticWPAdmin.tooltips
 */

(function($) {
	'use strict';

	window.AgenticWPAdmin = window.AgenticWPAdmin || {};

	/**
	 * Tooltips manager object.
	 *
	 * @type {Object}
	 */
	const Tooltips = {
		init: function() {
			this.bindEvents();
		},

		bindEvents: function() {
			// Click event to toggle tooltip
			$(document).on('click', '.agenticwp-tooltip-trigger', this.handleToggle.bind(this));

			// Close tooltip when clicking outside
			$(document).on('click', this.handleClickOutside.bind(this));

			// Close tooltip on ESC key
			$(document).on('keydown', this.handleEscape.bind(this));
		},

		handleToggle: function(e) {
			e.preventDefault();
			e.stopPropagation();

			const $trigger = $(e.currentTarget);
			// Support both old (.agenticwp-field-with-tooltip) and new (.agenticwp-field-wrapper) structures
			const $wrapper = $trigger.closest('.agenticwp-field-wrapper, .agenticwp-field-with-tooltip, .agenticwp-config-field');
			const $tooltip = $wrapper.find('.agenticwp-tooltip');
			const isVisible = !$tooltip.attr('hidden');

			// Close all other tooltips first
			this.closeAllTooltips();

			// Toggle current tooltip
			if (!isVisible) {
				this.showTooltip($trigger, $tooltip);
			}
		},

		showTooltip: function($trigger, $tooltip) {
			// Remove hidden to start the display
			$tooltip.removeAttr('hidden');
			$trigger.attr('aria-expanded', 'true');

			// Force a reflow to ensure the transition plays
			$tooltip[0].offsetHeight;

			// Add visible class to trigger fade-in and slide-up animation
			$tooltip.addClass('agenticwp-tooltip-visible');
		},

		hideTooltip: function($trigger, $tooltip) {
			// Remove visible class to trigger fade-out and slide-down animation
			$tooltip.removeClass('agenticwp-tooltip-visible');
			$trigger.attr('aria-expanded', 'false');

			// Wait for transition to complete before hiding
			setTimeout(function() {
				$tooltip.attr('hidden', 'hidden');
			}, 200); // Match the CSS transition duration
		},

		closeAllTooltips: function() {
			const self = this;
			$('.agenticwp-tooltip-visible').each(function() {
				const $tooltip = $(this);
				// Support both old and new structures
				const $wrapper = $tooltip.closest('.agenticwp-field-wrapper, .agenticwp-field-with-tooltip, .agenticwp-config-field');
				const $trigger = $wrapper.find('.agenticwp-tooltip-trigger');
				self.hideTooltip($trigger, $tooltip);
			});
		},

		handleClickOutside: function(e) {
			// If click is not on a tooltip trigger or inside a tooltip, close all tooltips
			// Support both old and new structures
			if (!$(e.target).closest('.agenticwp-field-wrapper, .agenticwp-field-with-tooltip, .agenticwp-config-field').length) {
				this.closeAllTooltips();
			}
		},

		handleEscape: function(e) {
			// ESC key closes tooltips
			if (e.key === 'Escape' || e.keyCode === 27) {
				this.closeAllTooltips();
			}
		}
	};

	window.AgenticWPAdmin.tooltips = Tooltips;

})(jQuery);
