<?php
/**
 * Chat Interface Handler
 * Manages chat interface integration and asset loading
 *
 * @package AgenticWP
 */

namespace Agentic_WP;

defined( 'ABSPATH' ) || exit;

/**
 * Chat Interface Handler.
 *
 * Manages chat interface integration and asset loading.
 *
 * @since 1.0.0
 */
class Chat_Interface {

	/**
	 * Initializes chat interface and registers hooks.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_chat_assets' ) );
		add_action( 'admin_footer', array( $this, 'render_chat_interface' ) );
		add_action( 'admin_bar_menu', array( $this, 'add_admin_bar_toggle' ), 9999 );
	}

	/**
	 * Enqueues CSS and JavaScript for chat panel.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function enqueue_chat_assets(): void {
		if ( ! is_admin() || wp_doing_ajax() ) {
			return;
		}

		$plugin_data = get_file_data(
			plugin_dir_path( __DIR__ ) . 'agenticwp.php',
			array( 'Version' => 'Version' )
		);
		$version     = $plugin_data['Version'] ?: '1.0.0';

		wp_enqueue_style(
			'agenticwp-chat-interface',
			plugin_dir_url( __DIR__ ) . 'assets/css/chat-interface.css',
			array(),
			$version
		);

		wp_enqueue_script(
			'dompurify',
			plugin_dir_url( __DIR__ ) . 'assets/js/vendor/dompurify.min.js',
			array(),
			$version,
			array(
				'strategy'  => 'defer',
				'in_footer' => true,
			)
		);

		wp_enqueue_script(
			'marked',
			plugin_dir_url( __DIR__ ) . 'assets/js/vendor/marked.min.js',
			array( 'dompurify' ),
			$version,
			array(
				'strategy'  => 'defer',
				'in_footer' => true,
			)
		);

		wp_enqueue_script(
			'agenticwp-chat-ui-helpers',
			plugin_dir_url( __DIR__ ) . 'assets/js/chat-ui-helpers.js',
			array( 'dompurify', 'marked' ),
			$version,
			array(
				'strategy'  => 'defer',
				'in_footer' => true,
			)
		);

		wp_enqueue_script(
			'agenticwp-chat-api',
			plugin_dir_url( __DIR__ ) . 'assets/js/chat-api.js',
			array( 'agenticwp-chat-ui-helpers' ),
			$version,
			array(
				'strategy'  => 'defer',
				'in_footer' => true,
			)
		);

		wp_enqueue_script(
			'agenticwp-toast-notifications',
			plugin_dir_url( __DIR__ ) . 'assets/js/toast-notifications.js',
			array(),
			$version,
			array(
				'strategy'  => 'defer',
				'in_footer' => true,
			)
		);

		wp_enqueue_script(
			'agenticwp-job-monitor',
			plugin_dir_url( __DIR__ ) . 'assets/js/job-completion-monitor.js',
			array( 'agenticwp-toast-notifications' ),
			$version,
			array(
				'strategy'  => 'defer',
				'in_footer' => true,
			)
		);

		wp_enqueue_script(
			'agenticwp-chat-interface',
			plugin_dir_url( __DIR__ ) . 'assets/js/chat-interface.js',
			array( 'agenticwp-chat-api', 'agenticwp-job-monitor' ),
			$version,
			array(
				'strategy'  => 'defer',
				'in_footer' => true,
			)
		);

		wp_localize_script(
			'agenticwp-chat-interface',
			'agenticwp_ajax',
			array(
				'ajax_url'         => admin_url( 'admin-ajax.php' ),
				'nonce'            => wp_create_nonce( 'agenticwp_chat_nonce' ),
				'job_status_nonce' => wp_create_nonce( 'agenticwp_job_status' ),
				'strings'          => array(
					'loading'         => __( 'Loading...', 'agenticwp' ),
					'sending'         => __( 'Sending...', 'agenticwp' ),
					'error'           => __( 'Something went wrong. Please try again.', 'agenticwp' ),
					'api_key_missing' => __( 'Please configure your OpenAI API key in Settings.', 'agenticwp' ),
				),
				'settings'         => array(
					'max_message_length'    => 1000,
					'message_history_limit' => Settings::get_message_history_limit(),
				),
			)
		);
	}

	/**
	 * Renders floating chat panel HTML.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function render_chat_interface(): void {
		if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$toast_enabled = Settings::get_option( 'agentic_wp_enable_toast_notifications', true );

		?>
		<script type="text/javascript">
		document.addEventListener('DOMContentLoaded', function() {
			if (typeof window.AgenticWPJobMonitor !== 'undefined') {
				window.agenticwpJobMonitor = new window.AgenticWPJobMonitor();
				window.agenticwpJobMonitor.setEnabled(<?php echo $toast_enabled ? 'true' : 'false'; ?>);
			}
		});
		</script>
		<?php
	}

	/**
	 * Adds chat toggle to admin bar.
	 *
	 * @since 1.0.0
	 *
	 * @param \WP_Admin_Bar $wp_admin_bar WordPress admin bar object.
	 * @return void
	 */
	public function add_admin_bar_toggle( $wp_admin_bar ): void {
		if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$wp_admin_bar->add_node(
			array(
				'id'     => 'agenticwp-chat',
				'parent' => 'top-secondary',
				'title'  => '<span class="agenticwp-admin-bar-icon">' . $this->get_chat_icon_svg() . '</span><span class="ab-label">AgenticWP</span>',
				'href'   => '#',
				'meta'   => array(
					'class' => 'agenticwp-admin-bar-toggle',
					'title' => __( 'Open AgenticWP Chat', 'agenticwp' ),
				),
			)
		);
	}

	/**
	 * Returns chat icon SVG markup.
	 *
	 * @since 1.0.0
	 *
	 * @return string SVG icon markup.
	 */
	private function get_chat_icon_svg(): string {
		return '<svg viewBox="0 0 24 24" width="20" height="20" aria-hidden="true" style="fill: currentColor; vertical-align: middle;"><path d="M20 2H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h14l4 4V4c0-1.1-.9-2-2-2zm-2 12H6v-2h12v2zm0-3H6V9h12v2zm0-3H6V6h12v2z"/></svg>';
	}
}